import { system, world, ItemStack, BlockPermutation } from "@minecraft/server";
import { ModalFormData, ActionFormData } from "@minecraft/server-ui";
import { itemApplyDamage, addItemOrSpawn, handleSitOnFurniture, decrementItemInHand, isCreative, spawnItemSilkTouch, getDirectionByPlayer, handleSitOld, addOrReplaceItem, toggleBlockState } from './util/utils';
import { airBlocks, blockException, colors, directions, solidBlocks, rightBlockLocation, leftBlockLocation, backBlockLocation, frontBlockLocation } from './util/globalVariables';
import { PianoCalm, PianoHaggstrom, PianoMiceOnVenus, PianoSubwooferLullaby, PianoSweden, PianoWetHands } from './DecoDrop/PianoSounds';
import { sitPet } from './util/decoScripts';

const DecoDropConfig = {
	accelerateTime: false,
	onlySleepAtNight: true
};
function extractColor(blockId) {
	const colorPattern = new RegExp(colors.join("|"), "i");
	const match = blockId.match(colorPattern);
	return match ? match[0] : 'black';
}

world.beforeEvents.worldInitialize.subscribe((e) => {
	e.blockComponentRegistry.registerCustomComponent("drop:cabinet", new DropCabinet());
	e.blockComponentRegistry.registerCustomComponent("drop:candle_table", new DropCandleTable());
	e.blockComponentRegistry.registerCustomComponent("drop:sit_on_chair", new DropSitOnChair());
	e.blockComponentRegistry.registerCustomComponent("drop:sit_chair", new DropSitOnChair2());
	e.blockComponentRegistry.registerCustomComponent("drop:fence_block", new DropFenceBlock());
	e.blockComponentRegistry.registerCustomComponent("drop:clay_beaker", new DropClayBeaker());
	e.blockComponentRegistry.registerCustomComponent("drop:dry_clay", new DropDryClay());
	e.blockComponentRegistry.registerCustomComponent("drop:lamp", new DropLamp());
	e.blockComponentRegistry.registerCustomComponent("drop:sit_wooden_stool", new DropSitWoodenStool());
	e.blockComponentRegistry.registerCustomComponent("drop:vase", new DropVase());
	e.blockComponentRegistry.registerCustomComponent("drop:pet_bed", new DropPetBed());
	e.blockComponentRegistry.registerCustomComponent("drop:hanging_pot", new DropHangingPot());
	e.blockComponentRegistry.registerCustomComponent("drop:bird_box", new DropBirdBox());
	e.blockComponentRegistry.registerCustomComponent("drop:add_sheet_music", new DropAddSheetMusic());
	e.blockComponentRegistry.registerCustomComponent("drop:piano_place", new DropPianoPlace());
	e.blockComponentRegistry.registerCustomComponent("drop:flatten_dough", new DropFlattenDough());
	e.blockComponentRegistry.registerCustomComponent("drop:break_dough", new DropBreakDough());
	e.blockComponentRegistry.registerCustomComponent("drop:rustic_oven", new DropRusticOven());
	e.blockComponentRegistry.registerCustomComponent("drop:add_ingredients", new DropAddIngredients());
	e.blockComponentRegistry.registerCustomComponent("drop:pizza_add_ingredients", new DropPizzaAddIngredients());
	e.blockComponentRegistry.registerCustomComponent("drop:sofa", new DropSofa());
	e.blockComponentRegistry.registerCustomComponent("drop:eat_pie", new DropEatPie());
	e.blockComponentRegistry.registerCustomComponent("drop:cut_block", new DropCutBlock());
	e.blockComponentRegistry.registerCustomComponent("drop:dyeable_cake", new DropDyeableCake());
	e.blockComponentRegistry.registerCustomComponent("drop:bowl", new DropBowl());
	e.blockComponentRegistry.registerCustomComponent("drop:frying_pan", new DropFryingPan());
	e.blockComponentRegistry.registerCustomComponent("drop:frying_pan_animation", new DropFryingPanAnimation());
	e.blockComponentRegistry.registerCustomComponent("drop:saucepan", new DropSaucepan());
	e.blockComponentRegistry.registerCustomComponent("drop:radio", new DropRadio());
	e.blockComponentRegistry.registerCustomComponent("drop:place_bed", new DropPlaceBed());
	e.blockComponentRegistry.registerCustomComponent("drop:bed_silk_touch", new DropBreakBed());
	e.blockComponentRegistry.registerCustomComponent("drop:sleep_bed", new DropBedSleep());
	e.blockComponentRegistry.registerCustomComponent("drop:sign_icon", new DropSignIcon());
	e.blockComponentRegistry.registerCustomComponent("drop:place_table_2x2", new DropPlaceTable2x2());
	e.blockComponentRegistry.registerCustomComponent("drop:table_silk_touch", new DropBreakTable2x2());
	e.blockComponentRegistry.registerCustomComponent("drop:ceramic_station", new DropCeramicStation());
	e.blockComponentRegistry.registerCustomComponent("drop:rotate_by_45", new DropRotateBy45());
	e.blockComponentRegistry.registerCustomComponent("drop:large_pot_add_dirt", new DropLargePotAddDirt());
	e.blockComponentRegistry.registerCustomComponent("drop:large_pot_plant", new DropLargePotPlant());
	e.blockComponentRegistry.registerCustomComponent("drop:break_flower", new DropBreakFlower());
	e.blockComponentRegistry.registerCustomComponent("drop:double_table_place", new DropDoubleTablePlace());
	e.blockComponentRegistry.registerCustomComponent("drop:break_main_st", new DropBreakMainBlock());
	e.blockComponentRegistry.registerCustomComponent("drop:break_left_st", new DropBreakLeftBlock());
	e.blockComponentRegistry.registerCustomComponent("drop:copper_teapot", new DropCopperTeapot());
	e.blockComponentRegistry.registerCustomComponent("drop:copper_teapot_consume", new DropCopperTeapotConsume());
	e.itemComponentRegistry.registerCustomComponent("drop:item_break_block", new DropItemBreakBlock());
	e.itemComponentRegistry.registerCustomComponent("drop:consume_golden_cake", new DropConsumeGoldenCake());
	e.itemComponentRegistry.registerCustomComponent("drop:get_cooking_recipe", new DropGetCookingRecipe());
	e.itemComponentRegistry.registerCustomComponent("drop:drink_mate", new DropDrinkMate());
	e.itemComponentRegistry.registerCustomComponent("drop:drink_milk", new DropDrinkMilk());
	e.itemComponentRegistry.registerCustomComponent("drop:start_config", new DropGlobalConfig());
});
class DropGlobalConfig {
	onUse(e) {
		const { source } = e;
		const config = world.getDynamicProperty("DecoDropConfig");
		const configObject = config ? { ...DecoDropConfig, ...JSON.parse(config) } : DecoDropConfig;
		const configUI = new ModalFormData()
			.title({ translate: 'drop.decodrop.settings' })
			.toggle({ translate: 'drop.decodrop.accelerate_time_when_sleeping' }, configObject.accelerateTime)
			.toggle({ translate: 'drop.decodrop.sleep_only_at_night' }, configObject.onlySleepAtNight);
		configUI.show(source).then(response => {
			if (response.canceled) return;
			const [accelerateTime, onlySleepAtNight] = response.formValues;
			configObject.accelerateTime = accelerateTime;
			configObject.onlySleepAtNight = onlySleepAtNight;
			world.setDynamicProperty("DecoDropConfig", JSON.stringify(configObject));
		});
	}
}
class DropBreakLeftBlock {
	onPlayerDestroy(e) {
		spawnItemSilkTouch(e, `${e.destroyedBlockPermutation.type.id.replace("left", "item")}`)
	}
}
class DropPianoPlace {
	beforeOnPlayerPlace(e) {
		const { player, block, permutationToPlace, dimension } = e;
		const direction = getDirectionByPlayer(player);
		const blockId = permutationToPlace.type.id;
		const rightBlock = block.offset(rightBlockLocation[direction]);
		if (airBlocks.includes(rightBlock.typeId)) {
			block.setPermutation(BlockPermutation.resolve(blockId.replace("item", "left"), { "minecraft:cardinal_direction": direction }));
			rightBlock.setPermutation(BlockPermutation.resolve(blockId.replace("item", "right"), { "minecraft:cardinal_direction": direction }));
			decrementItemInHand(player);
			const sound = blockId.includes("bamboo") ? "place.bamboo_wood" : blockId.includes("cherry") ? "place.cherry_wood" : blockId.includes("crimson") || blockId.includes("warped") ? "place.nether_wood" : "dig.wood";
			dimension.playSound(sound, block.center());
		}
		e.cancel = true;
	}
	onPlace(e) {
		const { block } = e;
		const direction = "north"
		const rightBlock = block.offset(rightBlockLocation[direction]);
		block.setPermutation(BlockPermutation.resolve(block.typeId.replace("item", "left"), { "minecraft:cardinal_direction": direction }));
		rightBlock.setPermutation(BlockPermutation.resolve(block.typeId.replace("item", "right"), { "minecraft:cardinal_direction": direction }));
	}
}
class DropAddSheetMusic {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const inventory = player.getComponent("inventory").container;
		const item = inventory.getItem(player.selectedSlotIndex);
		const hasSheetMusic = block.permutation.getState("drop:sheet");
		const direction = block.permutation.getState("minecraft:cardinal_direction");
		const pianoOffsets = {
			"drop.piano_left": {
				blockOffset: { north: { x: -1, y: 0, z: 0 }, south: { x: 1, y: 0, z: 0 }, west: { x: 0, y: 0, z: 1 }, east: { x: 0, y: 0, z: -1 } },
				entityOffset: { north: { x: 0, y: 0.5, z: 0.5 }, south: { x: 1, y: 0.5, z: 0.5 }, west: { x: 0.5, y: 0.5, z: 1 }, east: { x: 0.5, y: 0.5, z: 0 } }
			},
			"drop.piano_right": {
				blockOffset: { north: { x: 1, y: 0, z: 0 }, south: { x: -1, y: 0, z: 0 }, west: { x: 0, y: 0, z: -1 }, east: { x: 0, y: 0, z: 1 } },
				entityOffset: { north: { x: 1, y: 0.5, z: 0.5 }, south: { x: 0, y: 0.5, z: 0.5 }, west: { x: 0.5, y: 0.5, z: 0 }, east: { x: 0.5, y: 0.5, z: 1 } }
			}
		};
		const songMap = {
			"drop:sheet_calm": { soundName: "piano.calm", noteMapKey: PianoCalm },
			"drop:sheet_haggstrom": { soundName: "piano.haggstrom", noteMapKey: PianoHaggstrom },
			"drop:sheet_mice_on_venus": { soundName: "piano.mice_on_venus", noteMapKey: PianoMiceOnVenus },
			"drop:sheet_subwoofer_lullaby": { soundName: "piano.subwoofer_lullaby", noteMapKey: PianoSubwooferLullaby },
			"drop:sheet_sweden": { soundName: "piano.sweden", noteMapKey: PianoSweden },
			"drop:sheet_wet_hands": { soundName: "piano.wet_hands", noteMapKey: PianoWetHands }
		};
		const pianoType = block.hasTag("drop.piano_left") ? "drop.piano_left" : "drop.piano_right";
		const offsets = pianoOffsets[pianoType];
		function updateSheetState(state) {
			if (state) dimension.playSound("block.itemframe.add_item", block.center());
			else dimension.playSound("block.itemframe.remove_item", block.center());
			const oppositeBlock = block.offset(offsets.blockOffset[direction]);
			block.setPermutation(block.permutation.withState("drop:sheet", state));
			oppositeBlock.setPermutation(oppositeBlock.permutation.withState("drop:sheet", state));
		}
		function handleSongEnd(pianoEntity, dataSong) {
			player.onScreenDisplay.setActionBar([{ translate: 'drop.piano.song_ends'},{ translate: `${dataSong.song}`}]);
			dataSong.note = 0;
			pianoEntity.setDynamicProperty("drop:song", JSON.stringify(dataSong));
		}
		function playNotes(notesList, location) {
			notesList.forEach(noteInfo => {
				if (noteInfo.sound) dimension.playSound(noteInfo.sound, location, { pitch: noteInfo.pitch, volume: 1 });
			});
		}
		if (item && songMap[item.typeId] && !hasSheetMusic) {
			const PianoUI = new ActionFormData()
				.title({ translate: 'drop.piano.select_playback_mode' })
				.button({ translate: 'drop.piano.auto_mode' })
				.button({ translate: 'drop.piano.note_by_note' });
			PianoUI.show(player).then(response => {
				if (response.canceled) return;
				updateSheetState(true);
				decrementItemInHand(player);
				const pianoEntity = dimension.spawnEntity("drop:piano_entity", {
					x: block.location.x + offsets.entityOffset[direction].x,
					y: block.location.y + offsets.entityOffset[direction].y,
					z: block.location.z + offsets.entityOffset[direction].z
				});
				const songData = {
					song: item.typeId,
					note: 0,
					mode: response.selection === 0 ? "auto" : "manual",
					soundName: songMap[item.typeId].soundName
				};
				if (songData.soundName && response.selection === 0) {
					dimension.playSound(songData.soundName, pianoEntity.location);
					player.onScreenDisplay.setActionBar([{ translate: 'drop.piano.now_playing'},{ translate: `${songData.song}`}]);
				} else player.onScreenDisplay.setActionBar({ translate: `${songData.song}`});
				pianoEntity.setDynamicProperty("drop:song", JSON.stringify(songData));
			});
		} else if (hasSheetMusic) {
			const pianoEntity = dimension.getEntitiesAtBlockLocation(block.location).find(entity => entity.typeId === "drop:piano_entity");
			if (!pianoEntity) {
				updateSheetState(false);
				return;
			}
			const dataSong = JSON.parse(pianoEntity.getDynamicProperty("drop:song"));
			if (!dataSong) {
				updateSheetState(false);
				return;
			}
			if (dataSong.mode === "auto" && dataSong.soundName && player.isSneaking) {
				dimension.runCommandAsync(`stopsound @a ${dataSong.soundName}`);
				addItemOrSpawn(player, new ItemStack(dataSong.song, 1));
				updateSheetState(false);
				pianoEntity.remove();
			} else if (dataSong.mode === "manual") {
				if (player.isSneaking) {
					addItemOrSpawn(player, new ItemStack(dataSong.song, 1));
					updateSheetState(false);
					pianoEntity.remove();
				} else if (songMap[dataSong.song]) {
					const PianoSong = songMap[dataSong.song].noteMapKey;
					const notesList = PianoSong[dataSong.note];
					if (Array.isArray(notesList) && notesList.length > 0) {
						playNotes(notesList, pianoEntity.location);
						dataSong.note++;
						if (dataSong.note >= PianoSong.length) {
							handleSongEnd(pianoEntity, dataSong);
						} else {
							pianoEntity.setDynamicProperty("drop:song", JSON.stringify(dataSong));
							player.onScreenDisplay.setActionBar(`${dataSong.note} / ${PianoSong.length}`)
						}
					}
				}
			}
		}
	}
	onPlayerDestroy(e) {
		const { dimension, block } = e;
		const pianoEntity = dimension.getEntitiesAtBlockLocation(block.location).find(entity => entity.typeId === "drop:piano_entity");
		if (!pianoEntity) return;
		const dataSong = JSON.parse(pianoEntity.getDynamicProperty("drop:song"));
		if (!dataSong) return;
		dimension.runCommandAsync(`stopsound @a ${dataSong.soundName}`);
		dimension.spawnItem(new ItemStack(dataSong.song, 1), block.center());
	}
}
class DropDrinkMilk {
	onConsume(e) {
		const effects = ["nausea", "poison", "fatal_poison", "slowness", "weakness", "wither", "mining_fatigue", "levitation", "hunger", "darkness", "blindness", "bad_omen", "infested", "oozing", "weaving", "wind_charged"];
		effects.forEach(effect => e.source.removeEffect(effect));
	}
}
class DropDrinkMate {
	onConsume(e) {
		e.source.addEffect("regeneration", 100, { "amplifier": 0 });
	}
}
class DropCopperTeapotConsume {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const teapotEntity = dimension.getEntitiesAtBlockLocation(block.location).find(entity => entity.typeId === "drop:copper_teapot");
		if (!teapotEntity) {
			block.setPermutation(block.permutation.withState("drop:food", "empty"));
			return;
		}
		const item = player.getComponent("inventory").container.getItem(player.selectedSlotIndex);
		if (item?.typeId.includes("leaves") && block.permutation.getState("drop:food") === "water") {
			teapotEntity.setProperty("drop:icon", 2);
			block.setPermutation(block.permutation.withState("drop:food", "mate"));
			dimension.playSound("block.itemframe.add_item", block.center());
			decrementItemInHand(player);
			return;
		}
		if (item?.typeId !== "drop:cup_empty_item") return;
		const timer = teapotEntity.getProperty("drop:timer");
		if (timer >= 15) {
			const newItemType = teapotEntity.getProperty("drop:icon");
			const itemConvertsTo = ["drop:cup_water_item", "drop:cup_milk_item", "drop:cup_mate_item"];
			dimension.playSound("bottle.dragonbreath", block.center());
			addItemOrSpawn(player, new ItemStack(itemConvertsTo[newItemType], 1));
			decrementItemInHand(player);
			const quantity = teapotEntity.getDynamicProperty("drop:quantity") || 0;
			if (quantity > 0) {
				teapotEntity.setDynamicProperty("drop:quantity", quantity - 1);
			} else {
				block.setPermutation(block.permutation.withState("drop:food", "empty"));
				teapotEntity.remove();
			}
		}
	}
}
class DropCopperTeapot {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const item = player.getComponent("inventory").container.getItem(player.selectedSlotIndex);
		const allowedItems = {
			"minecraft:water_bucket": { itemConvertsTo: "minecraft:bucket", newState: "water", value: 0 },
			"minecraft:milk_bucket": { itemConvertsTo: "minecraft:bucket", newState: "milk", value: 1 }
		}
		if (!item) return;
		const convertBlock = allowedItems[item.typeId];
		if (convertBlock) {
			if (block.below().typeId !== "drop:rustic_oven") {
				player.onScreenDisplay.setActionBar({ translate: 'drop.copper_teapot.must_be_on_furnace' });
				return;
			}
			block.setPermutation(block.permutation.withState("drop:food", convertBlock.newState));
			dimension.playSound("bucket.empty_water", block.center())
			addOrReplaceItem(player, convertBlock.itemConvertsTo, 1);
			const teapotEntity = dimension.spawnEntity("drop:copper_teapot", { x: block.x + 0.5, y: block.y, z: block.z + 0.5 });
			teapotEntity.setProperty("drop:icon", convertBlock.value);
			teapotEntity.setDynamicProperty("drop:quantity", 2);
		}
	}
}
class DropBreakMainBlock{
	onPlayerDestroy(e) {
		spawnItemSilkTouch(e, `${e.destroyedBlockPermutation.type.id}_item`);
	}
}
class DropDoubleTablePlace {
	beforeOnPlayerPlace(e) {
		const { player, block, dimension, permutationToPlace } = e;
		const direction = getDirectionByPlayer(player);
		const rightBlock = block.offset(rightBlockLocation[direction]);
		if (airBlocks.includes(rightBlock.typeId)) {
			const color = extractColor(permutationToPlace.type.id);
			block.setPermutation(BlockPermutation.resolve(`drop:double_table_${color}`, {"minecraft:cardinal_direction": direction}));
			rightBlock.setPermutation(BlockPermutation.resolve(`drop:dt_${color}_right`, {"minecraft:cardinal_direction": direction}));
			decrementItemInHand(player);
			dimension.playSound("dig.wood", block.center());
		}
		e.cancel = true;
	}
	onPlace(e) {
		const { block } = e;
		const direction = "north";
		const rightBlock = block.offset(rightBlockLocation[direction]);
		const color = extractColor(block.typeId);
		block.setPermutation(BlockPermutation.resolve(`drop:double_table_${color}`, {"minecraft:cardinal_direction": direction}));
		rightBlock.setPermutation(BlockPermutation.resolve(`drop:dt_${color}_right`, {"minecraft:cardinal_direction": direction}));
	}
}
class DropBreakFlower {
	onPlayerDestroy(e) {
		spawnItemSilkTouch(e, e.destroyedBlockPermutation.type.id.replace("drop:flower_","minecraft:"));
	}
}
class DropLargePotAddDirt {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const item = player.getComponent("inventory").container.getItem(player.selectedSlotIndex);
		const allowedDirt = ["minecraft:grass_block", "minecraft:grass_path", "minecraft:dirt", "minecraft:coarse_dirt", "minecraft: dirt_with_roots", "minecraft:farmland", "minecraft:podzol"];
		if (allowedDirt.includes(item?.typeId)) {
			block.setType(`${block.typeId}_dirt`);
			dimension.playSound("block.composter.fill", block.center());
			decrementItemInHand(player);
		}
	}
}
class DropLargePotPlant {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const item = player.getComponent("inventory").container.getItem(player.selectedSlotIndex);
		const allowedFlowers = ["minecraft:dandelion", "minecraft:poppy", "minecraft:blue_orchid", "minecraft:allium", "minecraft:azure_bluet", "minecraft:red_tulip", "minecraft:orange_tulip", "minecraft:white_tulip", "minecraft:pink_tulip", "minecraft:oxeye_daisy", "minecraft:cornflower", "minecraft:lily_of_the_valley", "minecraft:torchflower", "minecraft:wither_rose", "minecraft:closed_eyeblossom", "minecraft:open_eyeblossom"];
		if (allowedFlowers.includes(item?.typeId) && airBlocks.includes(block.above().typeId)) {
			block.above().setType(item.typeId.replace("minecraft:", "drop:flower_"));
			dimension.playSound("dig.azalea_leaves", block.center());
			decrementItemInHand(player);
		}
	}
}
class DropSitOnChair2 {
	onPlayerInteract(e) {
		handleSitOnFurniture(e, "drop:chair_entity_drop", 0.4);
	}
}
class DropRotateBy45 {
	beforeOnPlayerPlace(e) {
		const { player } = e;
		const getBlockState = (rot) => {
			const normalized = rot + 180;
			const sector = Math.floor((normalized + 22.5) / 45) % 8;
			const directions = ["south", "south", "west", "west", "north", "north", "east", "east"];
			const rotates = [false, true, false, true, false, true, false, true];
			return { direction: directions[sector], rotate: rotates[sector] };
		};
		const yRot = player.getRotation().y;
		const { direction, rotate } = getBlockState(yRot);
		e.permutationToPlace = e.permutationToPlace.withState("minecraft:cardinal_direction", direction).withState("drop:rotate", rotate);
	}
}
class DropCeramicStation {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const item = player.getComponent("inventory").container.getItem(player.selectedSlotIndex);
		if (item?.typeId !== "minecraft:clay") {
			player.onScreenDisplay.setActionBar({ translate: 'drop.ceramic_station.use_clay_blocks' });
			return;
		}
		const CeramicUI = new ActionFormData()
			.title("DecoDrop Ceramics Station")
			.button("", "textures/ui/drop/clay_teapot")
			.button("", "textures/ui/drop/clay_pot")
			.button("", "textures/ui/drop/clay_beaker")
			.button("", "textures/ui/drop/clay_large_pot");
		CeramicUI.show(player).then(response => {
			if (response.canceled) return;
			const newItem = ["drop:clay_teapot", "drop:clay_pot", "drop:clay_beaker", "drop:clay_large_pot"][response.selection];
			const CeramicUIquantity = new ModalFormData()
				.title({ translate: 'drop.ceramic_station.select_item_quantity' })
				.slider({ translate: 'drop.ceramic_station.item_quantity' }, 1, item.amount, 1, 1);
			CeramicUIquantity.show(player).then(response => {
				if (response.canceled) return;
				const [ quantity ] = response.formValues;
				decrementItemInHand(player, false, quantity);
				dimension.spawnItem(new ItemStack(newItem, quantity), block.center());
			});
		});
	}
}
class DropBreakTable2x2 {
	onPlayerDestroy(e) {
		spawnItemSilkTouch(e, `${e.destroyedBlockPermutation.type.id.replace("_00","").replace("_01","").replace("_10","").replace("_11","")}`);
	}
}
class DropPlaceTable2x2 {
	beforeOnPlayerPlace(e) {
		const { player, block, dimension, permutationToPlace } = e;
		const direction = getDirectionByPlayer(player);
		const rightBlock = block.offset(rightBlockLocation[direction]);
		const backBlock = block.offset(backBlockLocation[direction]);
		const rightBackBlock = rightBlock.offset(backBlockLocation[direction]);
		const blockId = permutationToPlace.type.id;
		if ([rightBlock.typeId, backBlock.typeId, rightBackBlock.typeId].every(typeId => airBlocks.includes(typeId))) {
			const permutationsMap = {
				north: ["_00", "_01", "_11", "_10"],
				south: ["_10", "_11", "_01", "_00"],
				west: ["_01", "_10", "_00", "_11"],
				east: ["_11", "_00", "_10", "_01"]
			};
			const permutations = permutationsMap[direction];
			[block, rightBlock, backBlock, rightBackBlock].forEach((targetBlock, index) => {
				targetBlock.setPermutation(BlockPermutation.resolve(`${blockId}${permutations[index]}`));
			});
			decrementItemInHand(player);
			const sound = blockId.includes("bamboo") ? "place.bamboo_wood" : blockId.includes("cherry") ? "place.cherry_wood" : blockId.includes("crimson") || blockId.includes("warped") ? "place.nether_wood" : "dig.wood";
			dimension.playSound(sound, block.center());
		}
		e.cancel = true;
	}
}
let addTimeLoop = null;
let targetTime = null;

function startDayCycleAcceleration() {
	if (addTimeLoop !== null) return;
	addTimeLoop = system.runInterval(() => {
		if (targetTime === null) return;
		const players = world.getDimension("overworld").getPlayers();
		const anySleepingPlayer = players.some(player =>
			player.getComponent("minecraft:riding")?.entityRidingOn?.typeId === "drop:sleep_bed"
		);
		if (anySleepingPlayer) {
			const currentTime = world.getTimeOfDay();
			const newTime = (currentTime + 40) % 24000;
			world.setTimeOfDay(newTime);
			const timeDifference = (targetTime - currentTime + 24000) % 24000;
			if (timeDifference <= 40) {
				players[0].dimension.setWeather("Clear");
				world.setTimeOfDay(targetTime);
				stopDayCycleAcceleration();
				targetTime = null;
			}
		} else {
			stopDayCycleAcceleration();
			targetTime = null;
		}
	}, 1);
}
function stopDayCycleAcceleration() {
	if (addTimeLoop !== null) {
		system.clearRun(addTimeLoop);
		addTimeLoop = null;
	}
}
let animationLoop = null;
system.runInterval(() => {
	const players = world.getDimension("overworld").getPlayers();
	const sleepingPlayers = players.filter(player =>
		player.getComponent("minecraft:riding")?.entityRidingOn?.typeId === "drop:sleep_bed"
	);
	if (sleepingPlayers.length > 0) startAnimationLoop(sleepingPlayers);
	else stopAnimationLoop();
}, 20);
function startAnimationLoop(sleepingPlayers) {
	if (animationLoop !== null) return;
	sleepingPlayers.forEach(player => executeAnimation(player));
	animationLoop = system.runInterval(() => { sleepingPlayers.forEach(player => executeAnimation(player)); }, 20);
}
function stopAnimationLoop() {
	if (animationLoop !== null) {
		system.clearRun(animationLoop);
		animationLoop = null;
	}
}
function executeAnimation(player) {
	const block = player.dimension.getBlock(player.location);
	if (block?.hasTag("drop.bed")) {
		const direction = block.permutation.getState("minecraft:cardinal_direction");
		const animations = {
			north: "animation.drop.player.sleep_north",
			south: "animation.drop.player.sleep_south",
			west: "animation.drop.player.sleep_west",
			east: "animation.drop.player.sleep_east"
		};
		player.playAnimation(animations[direction], { stopExpression: "!q.is_riding" });
	}
}
class DropSignIcon {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const icon = block.permutation.getState("drop:icon");
		const newIcon = !player.isSneaking ? (icon + 1) % 10 : (icon - 1 + 10) % 10;
		block.setPermutation(block.permutation.withState("drop:icon", newIcon));
		dimension.playSound("block.itemframe.add_item", block.center());
	}
}
function PlayersSleep(player) {
	const config = world.getDynamicProperty("DecoDropConfig");
	const configObject = config ? { ...DecoDropConfig, ...JSON.parse(config) } : DecoDropConfig;
	if (configObject.onlySleepAtNight) {
		if (configObject.accelerateTime) {
			targetTime = 0;
			startDayCycleAcceleration();
		} else {
			player.camera.fade({
				fadeTime: { fadeInTime: 2.0, holdTime: 0.0, fadeOutTime: 1.0 },
				fadeColor: { red: 0.0, green: 0.0, blue: 0.0 },
			});
			system.runTimeout(() => {
				player.dimension.setWeather("Clear");
				world.setTimeOfDay(0);
				player.onScreenDisplay.setActionBar("06:00");
			}, 40);
		}
		return;
	}
	const SleepUI = new ActionFormData()
		.title({ translate: 'drop.bed.select_time_of_day' })
		.button({ translate: 'hostOption.time.sunrise' }, "textures/ui/drop/time_sunrise")     // 23000
		.button({ translate: 'hostOption.time.day' }, "textures/ui/drop/time_day")             // 1000
		.button({ translate: 'hostOption.time.noon' }, "textures/ui/drop/time_noon")           // 6000
		.button({ translate: 'hostOption.time.sunset' }, "textures/ui/drop/time_sunset")       // 12000
		.button({ translate: 'hostOption.time.night' }, "textures/ui/drop/time_night")         // 13000
		.button({ translate: 'hostOption.time.midnight' }, "textures/ui/drop/time_midnight");  // 18000
	SleepUI.show(player).then(response => {
		if (response.canceled) return;
		const selectedTime = [23000, 1000, 6000, 12000, 13000, 18000][response.selection];
		if (configObject.accelerateTime) {
			targetTime = selectedTime;
			startDayCycleAcceleration();
		} else {
			player.camera.fade({
				fadeTime: { fadeInTime: 2.0, holdTime: 0.0, fadeOutTime: 1.0 },
				fadeColor: { red: 0.0, green: 0.0, blue: 0.0 },
			});
			system.runTimeout(() => {
				player.dimension.setWeather("Clear");
				world.setTimeOfDay(selectedTime);
				const adjustedTime = (selectedTime + 6000) % 24000;
				const hours = Math.floor(adjustedTime / 1000);
				const minutes = Math.floor(((adjustedTime % 1000) / 1000) * 60);
				player.onScreenDisplay.setActionBar(`${hours.toString().padStart(2, '0')}:${minutes.toString().padStart(2, '0')}`);
			}, 40);
		}
	});
}
class DropBedSleep {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		if (dimension.getEntitiesAtBlockLocation(block.location).some(entity => entity.typeId === "drop:sleep_bed")) return;
		let center = block.center();
		const direction = block.permutation.getState("minecraft:cardinal_direction");
		if (block.hasTag("drop.bed_01")) {
			center = block.offset(leftBlockLocation[direction]).center();
		} else if (block.hasTag("drop.bed_10")) {
			center = block.offset(frontBlockLocation[direction]).center();
		} else if (block.hasTag("drop.bed_11")) {
			center = block.offset(leftBlockLocation[direction]).offset(frontBlockLocation[direction]).center();
		}
		switch (direction) {
			case "north": center.z += 0.5; center.x -= 0.5; break;
			case "south": center.z -= 0.5; center.x += 0.5; break;
			case "west":  center.x += 0.5; center.z += 0.5; break;
			case "east":  center.x -= 0.5; center.z -= 0.5; break;
		}
		if (dimension.id !== "minecraft:overworld") {
			dimension.createExplosion(center, 5, {breaksBlocks: true, causesFire: true});
			return;
		}
		const playerSpawnpoint = player.getSpawnPoint();
		const newSpawnpoint = { dimension: dimension, x: center.x, y: center.y + 0.5, z: center.z };
		const areSpawnPointsEqual = playerSpawnpoint?.dimension.id === newSpawnpoint.dimension.id && playerSpawnpoint?.x === newSpawnpoint.x && playerSpawnpoint?.y === newSpawnpoint.y && playerSpawnpoint?.z === newSpawnpoint.z;
		if (!areSpawnPointsEqual) {
			player.setSpawnPoint(newSpawnpoint);
			player.sendMessage({ rawtext: [{ text: '§i' }, { translate: 'tile.bed.respawnSet' }] });
		}
		const config = world.getDynamicProperty("DecoDropConfig");
		const configObject = config ? { ...DecoDropConfig, ...JSON.parse(config) } : DecoDropConfig;
		const time = world.getTimeOfDay();
		function placeBedAndSleep(player, dimension, center, direction) {
			const bedEntity = dimension.spawnEntity("drop:sleep_bed", center);
			bedEntity.setRotation({ x: 0, y: { north: 180, south: 0, west: 90, east: -90 }[direction] });
			bedEntity.getComponent("rideable").addRider(player);
			if (!addTimeLoop) PlayersSleep(player);
		}
		if (configObject.onlySleepAtNight) {
			if (time > 12500 && time < 23500) placeBedAndSleep(player, dimension, center, direction);
			else player.sendMessage({ rawtext: [{ text: '§i' }, { translate: 'tile.bed.noSleep' }] });
		} else {
			placeBedAndSleep(player, dimension, center, direction);
		}
	}
}
class DropBreakBed {
	onPlayerDestroy(e) {
		const color = extractColor(e.destroyedBlockPermutation.type.id);
		spawnItemSilkTouch(e, `drop:bed_2x2_${color}`);
	}
}
class DropPlaceBed {
	beforeOnPlayerPlace(e) {
		const { player, block, dimension, permutationToPlace } = e;
		const direction = getDirectionByPlayer(player);
		const rightBlock = block.offset(rightBlockLocation[direction]);
		const backBlock = block.offset(backBlockLocation[direction]);
		const rightBackBlock = rightBlock.offset(backBlockLocation[direction]);
		if ([rightBlock.typeId, backBlock.typeId, rightBackBlock.typeId].every(typeId => airBlocks.includes(typeId))) {
			const color = extractColor(permutationToPlace.type.id);
			block.setPermutation(BlockPermutation.resolve(`drop:bed_2x2_${color}_00`, {"minecraft:cardinal_direction": direction}));
			rightBlock.setPermutation(BlockPermutation.resolve(`drop:bed_2x2_${color}_01`, {"minecraft:cardinal_direction": direction}));
			backBlock.setPermutation(BlockPermutation.resolve(`drop:bed_2x2_${color}_10`, {"minecraft:cardinal_direction": direction}));
			rightBackBlock.setPermutation(BlockPermutation.resolve(`drop:bed_2x2_${color}_11`, {"minecraft:cardinal_direction": direction}));
			decrementItemInHand(player);
			dimension.playSound("dig.wood", block.center());
		}
		e.cancel = true;
	}
}
class DropItemBreakBlock {
	onMineBlock(e) {
		const { itemStack, source } = e;
		itemApplyDamage(source, itemStack);
	}
}
class DropConsumeGoldenCake {
	onConsume(e) {
		const { source } = e;
		source.addEffect("regeneration", 100, { "amplifier": 1 });
		source.addEffect("absorption", 2400, { "amplifier": 0 });
	}
}
class DropGetCookingRecipe{
	onUse(e) {
		const { source } = e;
		source.runCommandAsync("structure load drop.cooking_recipes ~~~ 0_degrees");
		decrementItemInHand(source);
	}
}
class DropCabinet {
	beforeOnPlayerPlace(e) {
		e.dimension.spawnEntity("drop:container_drop", e.block.center());
	}
	onPlayerInteract(e) {
		toggleBlockState(e, "drop:close", "block.barrel.open", "block.barrel.close");
	}
}
class DropCandleTable {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const item = player.getComponent("inventory").container.getItem(player.selectedSlotIndex);
		if (item?.typeId === "minecraft:flint_and_steel") {
			const permutation = block.permutation;
			if (!permutation.getState("drop:turn_on")) {
				block.setPermutation(permutation.withState("drop:turn_on", true));
				dimension.playSound("fire.ignite", block.location);
				itemApplyDamage(player, item);
			}
		}
	}
}
class DropSitOnChair {
	onPlayerInteract(e) {
		handleSitOld(e, "drop:chair_entity_drop", 0.56);
	};
}
class DropSitWoodenStool {
	onPlayerInteract(e) {
		handleSitOld(e, "drop:chair_entity_drop", 0.5);
	};
}
function updateBlockConnection(block, tag) {
	const states = block.permutation.getAllStates();
	const directionPrefixes = [":n", ":s", ":w", ":e"];
	const prefix = Object.keys(states).find(stateName => directionPrefixes.some(dir => stateName.includes(dir)))?.split(":")[0];
	if (!prefix) return;
	for (const direction in directions) {
		const offset = directions[direction];
		const adjacentBlock = block.offset(offset);
		if (!adjacentBlock) continue;
		const shouldConnect = adjacentBlock.hasTag(tag) || adjacentBlock.hasTag("mc.solid") || (solidBlocks.some(connect => adjacentBlock.typeId.includes(connect)) && !blockException.some(exception => adjacentBlock.typeId.includes(exception)) && adjacentBlock.typeId.includes("minecraft:"));
		const stateName = `${prefix}:${direction[0]}`;
		if (stateName in states) block.setPermutation(block.permutation.withState(stateName, shouldConnect ? 1 : 0));
	}
}
function updateAdjacentBlocksIfNeeded(block) {
	for (const direction in directions) {
		const offset = directions[direction];
		const adjacentBlock = block.offset(offset);
		if (adjacentBlock?.hasTag('mc.fence_block')) updateBlockConnection(adjacentBlock, 'mc.fence_block');
		if (adjacentBlock?.hasTag('mc.wall_block')) updateBlockConnection(adjacentBlock, 'mc.wall_block');
	}
}
world.afterEvents.playerPlaceBlock.subscribe((e) => {
	const { block } = e;
	updateAdjacentBlocksIfNeeded(block);
});
world.afterEvents.playerBreakBlock.subscribe((e) => {
	const { block } = e;
	updateAdjacentBlocksIfNeeded(block);
});
world.afterEvents.blockExplode.subscribe((e) => {
	const { block } = e;
	updateAdjacentBlocksIfNeeded(block);
});
class DropFenceBlock {
	onPlace(e) {
		const { block } = e;
		updateBlockConnection(block, 'mc.fence_block');
	}
}
class DropClayBeaker {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const quantity = block.permutation.getState("drop:quantity");
		const inventory = player.getComponent("inventory").container;
		const item = inventory.getItem(player.selectedSlotIndex);
		const updateBlock = (newQuantity) => {
			block.setPermutation(block.permutation.withState("drop:quantity", newQuantity));
			dimension.playSound("break.decorated_pot", block.location);
		};
		if (item === undefined || block.typeId !== item.typeId || quantity === 3) {
			if (quantity > 1 && quantity <= 3) {
				updateBlock(quantity - 1);
				addItemOrSpawn(player, new ItemStack(block.typeId, 1));
			}
		} else if (quantity < 3) {
			updateBlock(quantity + 1);
			if (!isCreative(player)) {
				decrementItemInHand(player);
			}
		}
	}
}
class DropDryClay {
	onRandomTick(e) {
		const { block, dimension } = e;
		if (block.below().typeId === "drop:rustic_oven") {
			const blockMapping = {
				"drop:clay_beaker": {id: "drop:terracotta_clay_beaker", rotable: true},
				"drop:clay_teapot": {id: "drop:terracotta_teapot", rotable: true},
				"drop:clay_pot": {id: "drop:terracotta_pot", rotable: false},
				"drop:clay_large_pot": {id: "drop:terracotta_large_pot", rotable: false}
			};
			const currentBlockType = block.typeId;
			const newBlockType = blockMapping[currentBlockType];
			if (!newBlockType) return;
			const newPermutation = newBlockType.rotable ? BlockPermutation.resolve(newBlockType.id, { "minecraft:cardinal_direction": block.permutation.getState("minecraft:cardinal_direction") }) : BlockPermutation.resolve(newBlockType.id);
			block.setPermutation(newPermutation);
			dimension.playSound("break.decorated_pot", block.center());
		}
	}
}
class DropLamp {
	onPlayerInteract(e) {
		toggleBlockState(e, "drop:state_on", "effect.lamp_on", "effect.lamp_off");
	}
}
class DropVase {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const inventory = player.getComponent("inventory").container;
		const item = inventory.getItem(player.selectedSlotIndex);
		const fluidLevel = block.permutation.getState("drop:fill");
		const fluidType = block.permutation.getState("drop:fluid");
		const fluidActions = {
			"minecraft:water_bucket": { fill: 11, type: "water", sound: "bucket.empty_water", replaceItem: "minecraft:bucket" },
			"minecraft:potion": { fill: 1, type: "water", sound: "bottle.dragonbreath", replaceItem: "minecraft:glass_bottle" },
			"minecraft:lava_bucket": { fill: 11, type: "lava", sound: "bucket.empty_lava", replaceItem: "minecraft:bucket" },
			"minecraft:honey_bottle": { fill: 1, type: "honey", sound: "bottle.dragonbreath", replaceItem: "minecraft:glass_bottle" },
			"minecraft:slime_ball": { fill: 1, type: "slime", sound: "step.slime", replaceItem: "minecraft:slime_ball" },
			"minecraft:milk_bucket": { fill: 11, type: "milk", sound: "bucket.empty_water", replaceItem: "minecraft:bucket" }
		};
		const bucketActions = {
			"water": { itemType: "minecraft:water_bucket", sound: "bucket.fill_water" },
			"lava": { itemType: "minecraft:lava_bucket", sound: "bucket.fill_lava" },
			"milk": { itemType: "minecraft:milk_bucket", sound: "bucket.fill_water" }
		};
		if (fluidLevel === 0 && fluidType === "none" && fluidActions[item?.typeId]) {
			const action = fluidActions[item.typeId];
			block.setPermutation(block.permutation.withState("drop:fill", action.fill).withState("drop:fluid", action.type));
			dimension.playSound(action.sound, block.location);
			if (item.typeId.includes("bucket")) {
				addOrReplaceItem(player, action.replaceItem, 1);
			} else {
				decrementItemInHand(player);
				if (item.typeId !== "minecraft:slime_ball") {
					addItemOrSpawn(player, new ItemStack(action.replaceItem, 1));
				}
			}
		}
		if (fluidLevel > 0 && fluidLevel < 11 && item) {
			const action = fluidActions[item.typeId];
			if (action && action.type === fluidType) {
				const newFluidLevel = Math.min(fluidLevel + (action.fill === 11 ? 11 : 1), 11);
				block.setPermutation(block.permutation.withState("drop:fill", newFluidLevel));
				dimension.playSound(action.sound, block.center());
				if (item.typeId.includes("bucket")) {
					addOrReplaceItem(player, action.replaceItem, 1);
				} else {
					decrementItemInHand(player);
					if (item.typeId !== "minecraft:slime_ball") {
						addItemOrSpawn(player, new ItemStack(action.replaceItem, 1));
					}
				}
			}
		}
		if (item?.typeId === "minecraft:glass_bottle" && fluidType !== "none") {
			const bottleTypeMap = {
				"water": { itemType: "minecraft:potion", fillSound: "bucket.fill_water", fillLevel: 1 },
				"honey": { itemType: "minecraft:honey_bottle", fillSound: "random.drink_honey", fillLevel: 1 }
			};
			const bottleAction = bottleTypeMap[fluidType];
			if (bottleAction && fluidLevel > 0) {
				const newFluidLevel = fluidLevel - bottleAction.fillLevel;
				block.setPermutation(block.permutation.withState("drop:fill", newFluidLevel));
				decrementItemInHand(player);
				if (item.typeId !== "minecraft:slime_ball") {
					addItemOrSpawn(player, new ItemStack(bottleAction.itemType, 1));
				}
				dimension.playSound("bottle.dragonbreath", block.location);
				dimension.playSound(bottleAction.fillSound, block.location);
				if (newFluidLevel === 0) {
					block.setPermutation(block.permutation.withState("drop:fill", 0).withState("drop:fluid", "none"));
				}
			}
		}
		if (item?.typeId === "minecraft:bucket" && fluidLevel === 11 && bucketActions[fluidType]) {
			const bucketAction = bucketActions[fluidType];
			addOrReplaceItem(player, bucketAction.itemType, 1);
			block.setPermutation(block.permutation.withState("drop:fill", 0).withState("drop:fluid", "none"));
			dimension.playSound(bucketAction.sound, block.location);
		}
		if (fluidType === "slime" && item?.typeId !== "minecraft:slime_ball") {
			dimension.playSound("step.slime", block.location);
			addItemOrSpawn(player, new ItemStack("minecraft:slime_ball", 1));
			if (fluidLevel > 1) block.setPermutation(block.permutation.withState("drop:fill", fluidLevel - 1));
			else block.setPermutation(block.permutation.withState("drop:fill", 0).withState("drop:fluid", "none"));
		}
	}
	onPlayerDestroy(e) {
		const { player, block, dimension, destroyedBlockPermutation } = e;
		if (isCreative(player)) return;
		const fluidLevel = destroyedBlockPermutation.getState("drop:fill");
		const fluidType = destroyedBlockPermutation.getState("drop:fluid");
		if (fluidLevel > 0) {
			if (fluidType === "slime") {
				dimension.spawnItem(new ItemStack("minecraft:slime_ball", fluidLevel), block.center());
			}
		}
	}
}
class DropPetBed {
	onPlayerInteract(e) {
		sitPet(e, ["minecraft:wolf", "minecraft:cat"], 0.25, 7);
	}
}
class DropBirdBox {
	onPlayerInteract(e) {
		sitPet(e, ["minecraft:parrot", "minecraft:chicken"], 0.2, 7);
	}
}
class DropHangingPot {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const inventory = player.getComponent("inventory").container;
		const item = inventory.getItem(player.selectedSlotIndex);
		const flowerState = block.permutation.getState("drop:flower");
		const allowedItems = ["minecraft:bone_meal", "minecraft:shears"];
		if (!allowedItems.includes(item?.typeId)) {
			player.onScreenDisplay.setActionBar({ translate: 'drop.hanging_pot.use_bone_meal' });
		}
		if (item?.typeId === "minecraft:bone_meal") {
			block.setPermutation(block.permutation.withState("drop:flower", Math.floor(Math.random() * 3) + 1));
			dimension.playSound("item.bone_meal.use", block.location);
			decrementItemInHand(player);
		} else if (flowerState > 0 && item?.typeId === "minecraft:shears") {
			block.setPermutation(block.permutation.withState("drop:flower", 0));
			dimension.playSound("mob.sheep.shear", block.location);
			if (!isCreative(player)) itemApplyDamage(player, item);
		}
	}
}
class DropFlattenDough {
	onPlayerInteract(e) {
		const { block, dimension } = e;

		const random = Math.random();
		if (random <= 0.25) {
			const match = block.typeId.match(/^(drop:dough_)(\d+)$/);
			const dough = match[1];
			const n = parseInt(match[2]);
			const newBlockType = `${dough}${n + 1}`;
			const newBlockPermutation = BlockPermutation.resolve(newBlockType);
			block.setPermutation(newBlockPermutation);
		}
		dimension.playSound("dig.honey_block", block.location);
	}
}
class DropBreakDough {
	onPlayerDestroy(e) {
		spawnItemSilkTouch(e, "drop:dough_item");
	}
}
class DropRusticOven{
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const item = player.getComponent("inventory").container.getItem(player.selectedSlotIndex);
		const permutation = block.permutation;
		const hasItem = permutation.getState("drop:has_item");
		const cookingItems = {
			"drop:apple_pie_dough_item": { value: 0, message: "drop.oven_apple_pie", to: "drop:apple_pie_block_item" },
			"drop:chocolate_pie_dough_item": { value: 1, message: "drop.oven_chocolate_pie", to: "drop:chocolate_pie_block_item" },
			"drop:pumpkin_pie_dough_item": { value: 2, message: "drop.oven_pumpkin_pie", to: "drop:pumpkin_pie_block_item" },
			"drop:pizza_dough_item": { value: 3, message: "drop.oven_pizza", to: "drop:pizza_item" },
			"drop:cake_dough_item": { value: 4, message: "drop.oven_cake", to: "drop:cake_base_item" }
		};
		if (!hasItem && item) {
			const selectedItem = cookingItems[item.typeId];
			if (!selectedItem) return;
			const ovenEntity = dimension.spawnEntity("drop:rustic_oven", { x: block.x + 0.5, y: block.y, z: block.z + 0.5 });
			ovenEntity.setProperty("drop:icon", selectedItem.value);
			player.onScreenDisplay.setActionBar({rawtext: [{ translate: selectedItem.message },{ translate: "drop.message.wait" }]});
			dimension.playSound("block.itemframe.add_item", block.center());
			block.setPermutation(permutation.withState("drop:has_item", true));
			decrementItemInHand(player);
		} else {
			const ovenEntity = dimension.getEntitiesAtBlockLocation(block.location).find(entity => entity.typeId === "drop:rustic_oven");
			if (!ovenEntity) {
				block.setPermutation(permutation.withState("drop:has_item", false));
				return;
			}
			const newItem = Object.values(cookingItems).find(item => item.value === ovenEntity.getProperty("drop:icon")).to;
			const timer = ovenEntity.getProperty("drop:timer");
			if (timer >= 15) {
				dimension.playSound("block.itemframe.remove_item", block.center());
				ovenEntity.remove();
				block.setPermutation(permutation.withState("drop:has_item", false));
				addItemOrSpawn(player, new ItemStack(newItem, 1));
			}
		}
	}
}
class DropAddIngredients {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const inventory = player.getComponent("inventory").container;
		const item = inventory.getItem(player.selectedSlotIndex);
		const itemToBlockMap = {
			"minecraft:apple": "drop:apple_pie_dough",
			"minecraft:cocoa_beans": "drop:chocolate_pie_dough",
			"minecraft:pumpkin": "drop:pumpkin_pie_dough",
			"minecraft:beetroot": "drop:beetroot_dough",
			"drop:dough_item": "drop:cake_dough"
		};
		if (item?.typeId in itemToBlockMap) {
			const newBlockType = itemToBlockMap[item.typeId];
			block.setPermutation(BlockPermutation.resolve(newBlockType));
			dimension.playSound("block.itemframe.add_item", block.location);
			decrementItemInHand(player);
		}
		if (item?.typeId.includes("drop:") && item?.typeId.includes("_knife")) {
			dimension.playSound("block.itemframe.remove_item", block.location);
			addItemOrSpawn(player, new ItemStack("drop:spaghetti", 3));
			block.setPermutation(BlockPermutation.resolve("minecraft:air"));
			itemApplyDamage(player, item);
		}
	}	
}
function UpdateSofa(block) {
	const permutation = block.permutation;
	const direction = permutation.getState("minecraft:cardinal_direction");
	const tag = `${block.typeId}_${direction}`;
	const rightBlock = block.offset(leftBlockLocation[direction]);
	const leftBlock = block.offset(rightBlockLocation[direction]);
	block.setPermutation(permutation.withState("drop:right", rightBlock.hasTag(tag)).withState("drop:left", leftBlock.hasTag(tag)));
}
function TestSofa(block, permutation) {
	let id = block.typeId;
	let direction = permutation.getState("minecraft:cardinal_direction");
	if (id !== "minecraft:air") UpdateSofa(block);
	else id = permutation.type.id;
	const tag = `${id}_${direction}`;
	const rightBlock = block.offset(leftBlockLocation[direction]);
	const leftBlock = block.offset(rightBlockLocation[direction]);
	[rightBlock, leftBlock].forEach(neighbor => { if (neighbor.hasTag(tag)) UpdateSofa(neighbor)});
}
class DropSofa {
	onPlayerInteract(e) {
		handleSitOld(e, "drop:chair_entity_drop", 0.4);
	};
	onPlace(e) {
		TestSofa(e.block, e.block.permutation);
	}
	onPlayerDestroy(e) {
		TestSofa(e.block, e.destroyedBlockPermutation);
	}
}
class DropPizzaAddIngredients {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const inventory = player.getComponent("inventory").container;
		const item = inventory.getItem(player.selectedSlotIndex);
		const states = block.permutation.getState("drop:states");
		if (item?.typeId === "drop:cheese_item" && states === "none") {
			dimension.playSound("block.itemframe.add_item", block.location);
			block.setPermutation(block.permutation.withState("drop:states", "cheese"));
			decrementItemInHand(player);
		}
		if (item?.typeId === "minecraft:porkchop" && states === "cheese") {
			dimension.playSound("block.itemframe.add_item", block.location);
			block.setPermutation(BlockPermutation.resolve("drop:pizza_dough"));
			decrementItemInHand(player);
		}
	}
}
class DropEatPie {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		removePiecesOrBlock(block, dimension);
		dimension.playSound("random.eat", block.center());
		player.addEffect("saturation", 1, { "amplifier": 3, "showParticles": false });
	}
}
class DropCutBlock {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const inventory = player.getComponent("inventory").container;
		const item = inventory.getItem(player.selectedSlotIndex);

		if (item?.typeId.includes("drop:") && item?.typeId.includes("_knife")) {
			const itemMap = {
				"drop:cake_base": "drop:cake_slice_base",
				"drop:cake_chocolate": "drop:cake_slice_chocolate",
				"drop:cake_gold": "drop:cake_slice_gold",
				"drop:cheese_block": "drop:cheese_item",
				"drop:pizza_block": "drop:pizza_slice_item"
			}
			addItemOrSpawn(player, new ItemStack(itemMap[block.typeId] || `${block.typeId}_item`, 1));
			removePiecesOrBlock(block, dimension);
			dimension.playSound("block.itemframe.remove_item", block.center());
			itemApplyDamage(player, item);
		}
	}
	onPlayerDestroy(e) {
		const blockId = e.destroyedBlockPermutation.type.id;
		const itemMap = {
			"drop:cake_base": "drop:cake_slice_base",
			"drop:cake_chocolate": "drop:cake_slice_chocolate",
			"drop:cake_gold": "drop:cake_slice_gold",
			"drop:cheese_block": "drop:cheese_item",
			"drop:pizza_block": "drop:pizza_slice_item"
		}
		spawnItemSilkTouch(e, itemMap[blockId] || `${blockId}_item`, 4 - e.destroyedBlockPermutation.getState("drop:pieces"));
	}
}
class DropDyeableCake {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const inventory = player.getComponent("inventory").container;
		const item = inventory.getItem(player.selectedSlotIndex);

		if (item?.typeId.includes("minecraft:") && item?.typeId.endsWith("_dye")) {
			const color = item.typeId.replace("minecraft:", "").replace("_dye", "");
			const blockColorMatch = block.typeId.match(/cake_(.*?)_(base|chocolate|gold)/);
			const blockColor = blockColorMatch ? blockColorMatch[1] : null;
			const isBaseBlock = block.typeId === "drop:cake_base" || block.typeId === "drop:cake_chocolate" || block.typeId === "drop:cake_gold";
			if (isBaseBlock || (blockColor && blockColor !== color)) {
				const direction = block.permutation.getState("minecraft:cardinal_direction");
				const pieces = block.permutation.getState("drop:pieces");
				const newBlockType = `drop:cake_${color}_${block.typeId.split("_").pop()}`;
				const newPermutation = BlockPermutation.resolve(newBlockType, {"minecraft:cardinal_direction": direction,"drop:pieces": pieces});
				block.setPermutation(newPermutation);
				dimension.playSound("block.itemframe.add_item", block.location);
				decrementItemInHand(player);
			}
		}
	}
}
system.afterEvents.scriptEventReceive.subscribe(e => {
	const { id, sourceEntity } = e;
	if (id !== "drop:bowl_convert_cheese") return;
	if (sourceEntity?.typeId !== "drop:bowl") return;
	const block = sourceEntity.dimension.getBlock(sourceEntity.location);
	if (block?.typeId === "drop:bowl_0_drop") {
		block.setPermutation(block.permutation.withState("drop:item", "cheese"));
	}
});
class DropBowl {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const item = player.getComponent("inventory").container.getItem(player.selectedSlotIndex);
		const state = block.permutation.getState("drop:item");
		if (state === "empty" && item?.typeId === "minecraft:milk_bucket") {
			block.setPermutation(block.permutation.withState("drop:item", "milk"));
			dimension.playSound("mob.mooshroom.suspicious_milk", block.location);
			addOrReplaceItem(player, "minecraft:bucket", 1);
		} else if (state === "milk" && item?.typeId === "minecraft:leather") {
			dimension.playSound("block.itemframe.add_item", block.location);
			player.onScreenDisplay.setActionBar({rawtext: [{ translate: 'drop.bowl.cheese_placed'}, { translate: 'drop.message.wait'}]});
			dimension.spawnEntity("drop:bowl", {x: block.x + 0.5, y: block.y, z: block.z + 0.5});
			decrementItemInHand(player);
		} else if (state === "cheese") {
			block.setPermutation(block.permutation.withState("drop:item", "empty"));
			dimension.playSound("block.itemframe.remove_item", block.location);
			dimension.getEntitiesAtBlockLocation(block.location).find(entity => entity.typeId === "drop:bowl")?.remove();
			addItemOrSpawn(player, new ItemStack("drop:cheese_block_item", 1));
		}
	}
}
class DropFryingPan {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const inventory = player.getComponent("inventory").container;
		const item = inventory.getItem(player.selectedSlotIndex);
		const hasItem = block.permutation.getState("drop:item");
		const direction = block.permutation.getState("minecraft:cardinal_direction");
		const x = block.x + 0.5;
		const z = block.z + 0.5;
		const rotationMap = {
			"south": { x: 0, y: 180 },
			"west": { x: 0, y: -90 },
			"east": { x: 0, y: 90 },
			"north": { x: 0, y: 0 }
		};
		if (!hasItem) {
			const eventMap = {
				"minecraft:beef": "drop:beef",
				"minecraft:chicken": "drop:chicken",
				"minecraft:cod": "drop:cod",
				"minecraft:kelp": "drop:kelp",
				"minecraft:mutton": "drop:mutton",
				"minecraft:porkchop": "drop:porkchop",
				"minecraft:potato": "drop:potato",
				"minecraft:rabbit": "drop:rabbit",
				"minecraft:salmon": "drop:salmon",
				"minecraft:egg": "drop:egg"
			};
			if (item?.typeId in eventMap) {
				const food = dimension.spawnEntity("drop:food_entity", { x: x, y: block.y + 0.15, z: z });
				food.setRotation(rotationMap[direction]);
				food.triggerEvent(eventMap[item.typeId]);
				block.setPermutation(block.permutation.withState("drop:item", true));
				dimension.playSound("block.itemframe.add_item", block.location);
				decrementItemInHand(player);
			}
		} else {
			const isAnimated = block.permutation.getState("drop:animation");
			const foodCooked = dimension.getEntitiesAtBlockLocation(block.location).find(entity => entity.typeId === "drop:food_entity");
			if (foodCooked?.getProperty("drop:cooked") && foodCooked?.getProperty("drop:flipped")) {
				const variant = foodCooked.getComponent("variant").value;
				dimension.playSound("block.itemframe.remove_item", block.location);
				const lootMap = {
					0: "minecraft:cooked_beef",
					1: "minecraft:cooked_chicken",
					2: "minecraft:cooked_cod",
					3: "minecraft:dried_kelp",
					4: "minecraft:cooked_mutton",
					5: "minecraft:cooked_porkchop",
					6: "minecraft:baked_potato",
					7: "minecraft:cooked_rabbit",
					8: "minecraft:cooked_salmon",
					9: "drop:fried_egg_d"
				};
				addItemOrSpawn(player, new ItemStack(lootMap[variant], 1));
				block.setPermutation(block.permutation.withState("drop:item", false));
				foodCooked.triggerEvent("drop:despawn");
			} else if (!isAnimated && block.offset({ x: 0, y: -1, z: 0 }).typeId === "drop:rustic_oven") {
				const fryingPan = dimension.spawnEntity("drop:frying_pan_entity", { x: x, y: block.y, z: z });
				fryingPan.setRotation(rotationMap[direction]);
	
				const foodEntity = dimension.getEntitiesAtBlockLocation(block.location).find(entity => entity.typeId === "drop:food_entity");
				if (foodEntity) {
					foodEntity.playAnimation("animation.item.food_entity");
					foodEntity.triggerEvent("drop:flip");
					block.setPermutation(block.permutation.withState("drop:animation", true));
				}
			}
		}
	}
}
class DropFryingPanAnimation{
	onTick(e) {
		const { block } = e;
		block.setPermutation(block.permutation.withState("drop:animation", false));
	}
}
class DropSaucepan {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const item = player.getComponent("inventory").container.getItem(player.selectedSlotIndex);
		if (!item) return;
		if (block.below().typeId !== "drop:rustic_oven") {
			player.onScreenDisplay.setActionBar({ translate: 'drop.saucepan.must_be_on_furnace' });
			return;
		}
		const food = block.permutation.getState("drop:items");
		const saucepanEntity = dimension.getEntitiesAtBlockLocation(block.location).find(entity => entity.typeId === "drop:saucepan");
		if (item.typeId === "minecraft:water_bucket" && food === "empty" && !saucepanEntity) {
			dimension.spawnEntity("drop:saucepan", { x: block.x + 0.5, y: block.y, z: block.z + 0.5 });
		}
		const conditions = {
			empty: { itemType: ["minecraft:water_bucket"], to: "water", sound: "bucket.empty_water", newItem: "minecraft:bucket" },
			water: { itemType: ["drop:spaghetti"], to: "water_spaghetti", sound: "block.itemframe.add_item" },
			water_spaghetti: {
				itemType: ["minecraft:beef", "minecraft:porkchop"],
				to: ["water_spaghetti_beef", "water_spaghetti_porkchop"],
				sound: "block.itemframe.add_item",
				actionbar: ["drop.water_spaghetti_beef", "drop.water_spaghetti_porkchop"]
			},
			water_spaghetti_beef: {
				itemType: ["minecraft:bowl"],
				to: "empty",
				sound: "block.itemframe.remove_item",
				newItem: "drop:spaghetti_beef",
				requiresSaucepan: true
			},
			water_spaghetti_porkchop: {
				itemType: ["minecraft:bowl"],
				to: "empty",
				sound: "block.itemframe.remove_item",
				newItem: "drop:spaghetti_porkchop",
				requiresSaucepan: true
			}
		};
		const current = conditions[food];
		if (current && current.itemType.includes(item.typeId)) {
			if (current.requiresSaucepan) {
				if (!saucepanEntity) {
					block.setPermutation(block.permutation.withState("drop:items", "empty"));
					return;
				}
				saucepanEntity.setProperty("drop:icon", 1);
				if (saucepanEntity.getProperty("drop:timer") !== 15) return;
				saucepanEntity.remove();
			}
			const index = current.itemType.indexOf(item.typeId);
			const nextState = Array.isArray(current.to) ? current.to[index] : current.to;
			const actionbar = Array.isArray(current.actionbar) ? current.actionbar[index] : current.actionbar;
			block.setPermutation(block.permutation.withState("drop:items", nextState));
			dimension.playSound(current.sound, block.center());
			if (actionbar) player.onScreenDisplay.setActionBar({translate: actionbar});
			if (current.newItem) addOrReplaceItem(player, current.newItem, 1);
			else {
				if (food === "water_spaghetti") {
					saucepanEntity?.setProperty("drop:icon", index);
					saucepanEntity?.setProperty("drop:icon_visibility", true)
				};
				decrementItemInHand(player);
			};
		}
	}
}
class DropRadio {
	onPlayerInteract(e) {
		const { player, block, dimension } = e;
		const sounds = [
			{ id: "record.13", texture: "textures/items/record_13" },
			{ id: "record.cat", texture: "textures/items/record_cat" },
			{ id: "record.blocks", texture: "textures/items/record_blocks" },
			{ id: "record.chirp", texture: "textures/items/record_chirp" },
			{ id: "record.far", texture: "textures/items/record_far" },
			{ id: "record.mall", texture: "textures/items/record_mall" },
			{ id: "record.mellohi", texture: "textures/items/record_mellohi" },
			{ id: "record.stal", texture: "textures/items/record_stal" },
			{ id: "record.strad", texture: "textures/items/record_strad" },
			{ id: "record.ward", texture: "textures/items/record_ward" },
			{ id: "record.11", texture: "textures/items/record_11" },
			{ id: "record.wait", texture: "textures/items/record_wait" },
			{ id: "record.otherside", texture: "textures/items/record_otherside" },
			{ id: "record.5", texture: "textures/items/record_5" },
			{ id: "record.pigstep", texture: "textures/items/record_pigstep" },
			{ id: "record.relic", texture: "textures/items/music_disc_relic" },
			{ id: "record.creator", texture: "textures/items/music_disc_creator" },
			{ id: "record.creator_music_box", texture: "textures/items/music_disc_creator_music_box" },
			{ id: "record.precipice", texture: "textures/items/music_disc_precipice" },
		];
		function showRadioUI(player, radioEntity) {
			const radioUI = new ActionFormData().title({ translate: 'drop.radio.sound_list' });
			sounds.forEach(sound => {
				radioUI.button({ rawtext: [{ text: '§l' }, { translate: `item.${sound.id.replace(".", "_")}.desc` }] }, sound.texture);
			});
			radioUI.show(player).then(response => {
				if (response.canceled) return;
				const selectedSound = sounds[response.selection].id;
				const oldSound = radioEntity?.getDynamicProperty("drop:sound");
				if (oldSound) {
					player.runCommandAsync(`stopsound @a[r=20] ${oldSound}`);
				}
				if (radioEntity) {
					radioEntity.setDynamicProperty("drop:sound", selectedSound);
				} else {
					const newRadio = dimension.spawnEntity("drop:radio_entity", block.center());
					newRadio.setDynamicProperty("drop:sound", selectedSound);
				}
				dimension.playSound(selectedSound, block.center());
				player.onScreenDisplay.setActionBar({ translate: `item.${selectedSound}.drop` });
			});
		}
		const radioEntity = dimension.getEntitiesAtBlockLocation(block.location).find(entity => entity.typeId === "drop:radio_entity");
		if (player.isSneaking && radioEntity) {
			const oldSound = radioEntity.getDynamicProperty("drop:sound");
			if (oldSound) player.runCommandAsync(`stopsound @a[r=20] ${oldSound}`);
			dimension.playSound("block.itemframe.remove_item", block.center());
			radioEntity.remove();
		} else {
			showRadioUI(player, radioEntity);
		}
	}
}
system.afterEvents.scriptEventReceive.subscribe(e => {
	const { id, sourceEntity } = e;
	if (id !== "drop:radio_stopsound") return;
	if (sourceEntity?.typeId !== "drop:radio_entity") return;
	const sound = sourceEntity.getDynamicProperty("drop:sound");
	if (!sound) return;
	const block = sourceEntity.dimension.getBlock(sourceEntity.location);
	block.dimension.runCommandAsync(`stopsound @a ${sound}`);
});
function removePiecesOrBlock(block) {
	const pieces = block.permutation.getState("drop:pieces");
	if (pieces < 3) block.setPermutation(block.permutation.withState("drop:pieces", pieces + 1));
	else block.setPermutation(BlockPermutation.resolve("minecraft:air"));
};